/* -*-C-*-
 ##############################################################################
 #
 # File:        examples/srcarb_intr2.c
 # RCS:         $Id: srcarb_intr2.c,v 1.10 1999/03/25 00:29:55 ericjb Exp $
 # Description: Host program for arb source 
 # Created:     July 15, 1997
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This is a host program that will demonstrate operation of the 
 # continous arb source mode.  It uses interrupts (default) or polling to
 # check if the E1434 (or source option) is ready for more arb data.
 # The interrupt handler also checks for shutdown, overload, underrun,
 # and overread conditions.  This program works with up to 5 source channels
 # on the same module.
 #
 # The equipment required for this demo is:
 #    1. VXI card cage with MXI interface to host.
 #    2. One E1434 (or E1432/33 with source option).
 #   
 # Set the logical address of the module to 8 (default), or use the
 # command line argument -L<addr> to specify the logical address.
 #
 # Hook up a scope to the source output connector.
 #
 # To run the measurement:
 #    start the program (include appropriate command line options
 #                       as desired).
 #
 # Valid command line options are:
 #   -b <arb source buffer size>
 #   -c <number of source chans on>
 #   -f <clock freq>
 #   -p Use polling instead of interrupts
 #   -s <span>
 #   -v verbose
 #   -L <Logical address>
 #
 # Note:  If overread conditions are reported, try using a lower span
 #        (or fewer chans) so that the host computer can service the 
 #        arb data transfers fast enough to keep up with the source output 
 #        rate.
 #  
 # Revisions:
 #
 ##############################################################################
 */

#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include <sicl.h>		/* For SICL functions */
#include <math.h>
#include "e1432.h"

/*
#define DPRINT
*/

#define	BLOCKSIZE	1024
#define	CHAN_MAX	32
#define	SRC_MAX		5

#define SIGLENGTH	65536
#define SIG_LENGTH_MAX  131072
#define SIGPERIOD	256
#define XFRLENGTH	4096
#define XFRLENGTHMAX	4096
#define SRCBUFRSIZE	4096
#define SRCBUFRSIZEMAX	40960

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

#define GET_INT(parm, msg)                      \
{                                               \
    rtn = sscanf(optarg, "%f", &ftmp);          \
    if ( rtn == 1 )                             \
    {                                           \
        (parm) = (LONGSIZ32)(ftmp + .5);        \
    }                                           \
    else                                        \
    {                                           \
        (void) fprintf(stderr, msg, optarg);    \
        exit(2);                                \
    }                                           \
}

#define GET_FLOAT(parm, msg)                    \
{                                               \
    rtn = sscanf(optarg, "%f", &(parm));        \
    if ( rtn != 1 )                             \
    {                                           \
        (void) fprintf(stderr, msg, optarg);    \
        exit(2);                                \
    }                                           \
}


/* Make these global so the interrupt handler can get to them easily */
static E1432ID hw;
static SHORTSIZ16 chan_list[CHAN_MAX];
static SHORTSIZ16 group;
static int chan_count;
static int verbose = 0;

static SHORTSIZ16 src_list[SRC_MAX];
static SHORTSIZ16 src_group;
static int src_count;

static SHORTSIZ16 global_list[CHAN_MAX + SRC_MAX];
static SHORTSIZ16 global_group;
static int global_count;

static INST irq_id;
static int irq_laddr;
static int irq_flag;

/******************* arb source routines **************************/

static LONGSIZ32 src_buff[E1432_SRC_DATA_NUMWORDS_MAX];

static LONGSIZ32 src_sigbuff1_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff1_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_b[SIG_LENGTH_MAX];

static LONGSIZ32 arb_siglength;
static LONGSIZ32 arb_sigperiod;
static SHORTSIZ16 arb_xfrmode, arb_xfrmodewait;

static SHORTSIZ16 source_mode, arb_mode;
static LONGSIZ32 arb_srcbufsize;
static LONGSIZ32 arb_xfrsize;


static LONGSIZ32 src_sigbuff[SIG_LENGTH_MAX];
static LONGSIZ32 *src_sigbuffptr;
static LONGSIZ32 *src_siglast;
static SHORTSIZ16 sigtype;

static int rd4data_set_count, rd4data_clr_count;

static LONGSIZ32 *src_sigbuffptrs_a[5];
static LONGSIZ32 *src_sigbuffptrs_b[5];
static LONGSIZ32 arb_xfrsize2, arb_srcbufleft;

void
src_sig(LONGSIZ32 siglength, int sigperiod, long *sigbuff)
{
    int i; 

    for(i=0; i<siglength; i++)
    {
	sigbuff[i] = (long)((float) 0x7fffffffL *
			sin(2.0*M_PI*(float)i/((float)sigperiod)));
    }
}

/******************* interrrupt routines **************************/

static int
read_data(void)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    int     i;

    /* Read some data */

    for (i = 0; i < chan_count; i++)
    {
/*
   (void) printf("irq handler input channel %d, chan_list[i](%x)\n", i,chan_list[i]);
 */
	CHECK(e1432_read_float64_data(hw, chan_list[i],
				      E1432_TIME_DATA, buffer,
				      BLOCKSIZE, NULL, &count));
/*
   (void) printf("channel %dd    data points read =  %d\n", i , count);
 */
	if (count != BLOCKSIZE)
	{
	    DEBUG((void) printf("Actual count was %d\n", count));
	    return -1;
	}
    }

    return 0;
}

static int
srcstatus(void)
{
    int     i, error;

    /* 
       Only one interrupt per measurement for src_shutdown and
       src_overload.  These src status conditions are latched until 
       the source is turned off, then cleared. The source status irq 
       is reenabled when the source is turned on.  

       Multiple interrupts per measurement for src_overread and
       src_underrun.  These src status conditions are latched until
       a call to the associated check_src_ function (at which time
       they are cleared).  Interrupts will not occur again for the
       same src status condition until that status condition has
       been cleared by a call to the associated check_src_ function.
       */


    /* check source status */
    for (i = 0; i < src_count; i++)
    {
	if (e1432_check_src_arbrdy(hw, src_list[i], arb_xfrmode) == 1)
	{
	    if(verbose) printf("    src_arbrdy from %d\n", src_list[i]);
	    /* down load signal */
            error = e1432_write_srcbuffer_data(hw, src_list[i],
		src_sigbuffptrs_a[i], (SHORTSIZ16)  arb_xfrsize, arb_xfrmode);
            if(error)
               printf("Error %d in e1432_write_srcbuffer_data\n", error);
	}

	if (e1432_check_src_shutdown(hw, src_list[i]) == 1)
	    (void) printf("    src_shutdown from  %d\n", src_list[i]);
	if (e1432_check_src_overload(hw, src_list[i]) == 1)
	    (void) printf("    src_overload from  %d\n", src_list[i]);
	if (e1432_check_src_overread(hw, src_list[i]) == 1)
	    (void) printf("    src_overread from %d\n", src_list[i]);
	if (e1432_check_src_underrun(hw, src_list[i]) == 1)
	    (void) printf("    src_underrun from %d\n", src_list[i]);
    }

    return 0;
}

#ifdef DPRINT2
static void
irqstatus(void)
{
    LONGSIZ32 status;
    LONGSIZ32 xfrbuf_wds, srcbuf_state_AB, srcbuf_state_A, srcbuf_state_B;

    /* check irq status */

    if (src_count > 0)
    {
	e1432_read32_register(hw, src_list[0], E1432_IRQ_STATUS2_REG, &status);
	(void) printf("E1432_IRQ_STATUS2_REG = 0x%x\n", status);
	e1432_read32_register(hw, src_list[0], E1432_SRC_STATUS_REG, &status);
	(void) printf("E1432_SRC_STATUS_REG = 0x%x\n", status);
    }
    else
    {
	if (chan_count > 0)
	{
	    e1432_read32_register(hw, chan_list[0], E1432_IRQ_STATUS2_REG,
					&status);
	    (void) printf("    E1432_IRQ_STATUS2_REG = 0x%x\n\n",
			  status);
	}
    }

    (void) e1432_get_src_arbstates(hw, src_list[0], &xfrbuf_wds,
				   &srcbuf_state_AB, &srcbuf_state_A,
				   &srcbuf_state_B);
    (void) printf("xfrbuf_wds = %d, srcbuf_state_AB = 0x%x, "
		  "srcbuf_state_A = 0x%x, srcbuf_state_B = 0x%x\n",
		  xfrbuf_wds, srcbuf_state_AB,
		  srcbuf_state_A, srcbuf_state_B);

}
#endif

static void
irq_handler(INST id, long reason, long sec)
{
#ifdef DPRINT
    DEBUG((void) printf("\nirq_handler called: sec = 0x%x\n", sec));
#endif

    if (id != irq_id)
	(void) printf("Error: irq_handler got wrong id\n");
    if (reason != I_INTR_VXI_SIGNAL)
	(void) printf("Error: irq_handler got wrong reason\n");
    if ((sec & E1432_IRQ_STATUS_LADDR_MASK) != irq_laddr)
	(void) printf("Error: irq_handler got wrong laddr\n");

    irq_flag = 1;

    /* print out status */
#ifdef DPRINT2
    irqstatus();
#endif

    if ((sec & E1432_IRQ_SRC_STATUS) != 0)
	(void) srcstatus();
    if ((sec & E1432_IRQ_BLOCK_READY) != 0)
	(void) read_data();
    if ((sec & (E1432_IRQ_BLOCK_READY | E1432_IRQ_SRC_STATUS)) == 0)
	(void) printf("Error: irq_handler got wrong sec\n");

    /* reset interrupt */

    if (src_count > 0)
    {
	if(verbose) printf("reenable_interrupts\n");
	(void) e1432_reenable_interrupt(hw, src_group);
    }
    else if (chan_count > 0)
	(void) e1432_reenable_interrupt(hw, group);
}


static int
irq_setup(int laddr, int *line)
{
    struct vxiinfo info;
    unsigned long slot0_laddr;
    char    addr[16];
    INST    id;
    int     i;

    /* Get the interrupt line to use */
    id = iopen("vxi");
    if (id == 0)
    {
	DEBUG((void) printf("Error: iopen returned %d\n", id));
	return -1;
    }
    CHECK(ivxibusstatus(id, I_VXI_BUS_LADDR, &slot0_laddr));
    CHECK(ivxirminfo(id, slot0_laddr, &info));
    for (i = 0; i < 8; i++)
	if (info.int_handler[i] != -1)
	{
	    *line = info.int_handler[i];
	    break;
	}
    if (i == 8)
    {
	DEBUG((void) printf("Error: no interrupt lines available\n"));
	return -1;
    }
    DEBUG((void) printf("Using VME interrupt line %d\n", *line));

    /* Set up the interrupt handler routine */
    irq_laddr = laddr;
    (void) sprintf(addr, "vxi,%d", irq_laddr);
    irq_id = iopen(addr);
    if (irq_id == 0)
    {
	DEBUG((void) printf("Error: iopen returned %d\n", irq_id));
	return -1;
    }
    CHECK(ionintr(irq_id, irq_handler));
    CHECK(isetintr(irq_id, I_INTR_VXI_SIGNAL, 1));

    return 0;
}

/**********************************************************************/

int
main(int argn, char **argv)
{
    int     i, k, line,error;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 laddr = 8;
    int opt, rtn;
    float ftmp;
    LONGSIZ32 tmp = 0;
    int src_chans = 0;
    int arb_polling = 0;
    FLOATSIZ32 src_span = 0, src_clock = 51200;

    src_sigbuffptrs_a[0]= &src_sigbuff1_a[0];
    src_sigbuffptrs_a[1]= &src_sigbuff2_a[0];
    src_sigbuffptrs_a[2]= &src_sigbuff3_a[0];
    src_sigbuffptrs_a[3]= &src_sigbuff4_a[0];
    src_sigbuffptrs_a[4]= &src_sigbuff5_a[0];
    src_sigbuffptrs_b[0]= &src_sigbuff1_b[0];
    src_sigbuffptrs_b[1]= &src_sigbuff2_b[0];
    src_sigbuffptrs_b[2]= &src_sigbuff3_b[0];
    src_sigbuffptrs_b[3]= &src_sigbuff4_b[0];
    src_sigbuffptrs_b[4]= &src_sigbuff5_b[0];
    
    /* Initialize the library */
    CHECK(e1432_init_io_driver());

    /* Change this 0 to 1 to see call tracing */
    e1432_trace_level(0);

    arb_srcbufsize = SRCBUFRSIZE;

    while (( opt = getopt(argn, argv, "b:c:f:ps:vL:")) != EOF)
    {
       switch (opt)
       {
	  case 'b':
	     GET_INT(tmp, " Source buffer size %s not converted");
             arb_srcbufsize = tmp;
	     break;
	  case 'c':
	     GET_INT(tmp, " Logical Address %s not converted");
             src_chans = tmp;
	     break;
	  case 'f':
	     GET_FLOAT(ftmp, " clock freq %s not converted");
             src_clock = ftmp;
	     break;
	  case 'p':
             arb_polling = 1;
	     break;
	  case 's':
	     GET_FLOAT(ftmp, " source span %s not converted");
             src_span = ftmp;
	     break;
	  case 'v':
	     verbose = 1;
	     break;
	  case 'L':
	     GET_INT(tmp, " Logical Address %s not converted");
             laddr = tmp;
	     break;
	  default:
	     printf("%s usage:\n", argv[0]);
	     printf("    -b source buffer size\n");
	     printf("    -c number of source chans on\n");
	     printf("    -f clock freq\n");
	     printf("    -p Use polling instead of interrupts\n");
	     printf("    -s span\n");
	     printf("    -v verbose\n");
	     printf("    -L <Logical address>\n");
	     exit(0);
       }

    }
 
 
    /* Use e1432_get_hwconfig to see if the module already has
       firmware.  If this errors, assume we need to install firmware,
       so use e1432_install to do it. */
    error = e1432_get_hwconfig(1, &laddr, &hwconfig);
    if (error != 0)
    {
	(void) printf("downloading /opt/e1432/lib/sema.bin\n");
        error = e1432_install(1, &laddr, 0,
                              "/opt/e1432/lib/sema.bin");
        if (error)
	{
            (void) printf("e1432_install failed, error: %d\n", error);
	    return -1;
	}
	
	CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));
    }

    /* Turn on debugging prints, a good idea while developing.  But do
       it after the above e1432_get_hwconfig, because we know that
       might error if the firmware is not yet installed in the
       module. */
    (void) e1432_print_errors(1);

    e1432_debug_level(0);

    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));

    /* Create channel group */
    if (hwconfig.input_chans > 0)
    {
	chan_count = hwconfig.input_chans;
	if (chan_count > CHAN_MAX)
	    chan_count = CHAN_MAX;
	for (i = 0; i < chan_count; i++)
	    chan_list[i] = E1432_INPUT_CHAN(i + 1);
	group = e1432_create_channel_group(hw, chan_count, chan_list);
	if (group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				group));
	    return -1;
	}
    }
    else
	chan_count = 0;

    /* Create source group */
    if (hwconfig.source_chans > 0)
    {
	src_count = hwconfig.source_chans;
	if (src_count > SRC_MAX)
	    src_count = SRC_MAX;
	if (src_chans)
        {
	    src_count = src_chans;
	    if(verbose)
	      printf("src_chans = %d, src_count = %d\n", src_chans, src_count); 
        }
	for (i = 0; i < src_count; i++)
	    src_list[i] = E1432_SOURCE_CHAN(i + 1);
	src_group = e1432_create_channel_group(hw, src_count, src_list);
	if (src_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				src_group));
	    return -1;
	}
    }
    else
	src_count = 0;

    /* Create global group  of all input and source channels */

    global_count = 0;

    for (i = 0; i < chan_count; i++)
    {
	global_list[global_count] = chan_list[i];
	global_count = global_count + 1;
    }
    for (i = 0; i < src_count; i++)
    {
	global_list[global_count] = src_list[i];
	global_count = global_count + 1;
    }

    if (global_count > 0)
	global_group = e1432_create_channel_group(hw, global_count,
						  global_list);

    (void) printf("# input channels = %d, # source channels = %d\n",
		  chan_count, src_count);
    (void) printf("# total channels = %d\n", global_count);


/* initialize source */
    if (src_count > 0)
    {
	CHECK(e1432_set_interrupt_mask(hw, src_group, 0));
        CHECK(e1432_set_clock_freq(hw, src_group, src_clock));
        if(src_span) 
           CHECK(e1432_set_span(hw, src_group, src_span));
	else
           CHECK(e1432_set_span(hw, src_group, src_clock/(float)2.56));
        CHECK(e1432_get_span(hw, src_group, &src_span));
	printf("span = %f\n", src_span);

	source_mode = E1432_SOURCE_MODE_ARB;
	arb_mode = E1432_SRCBUFFER_CONTINUOUS;

	arb_siglength = SIGLENGTH;
	arb_sigperiod = SIGPERIOD;
        arb_xfrsize = XFRLENGTH;

	arb_xfrmode = E1432_SRC_DATA_MODE_AB;
	arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITAB;

	(void) printf("arb_xfrsize %d, arb_srcbufsize %d, arb_siglength %d\n",
		      arb_xfrsize, arb_srcbufsize, arb_siglength);
	(void) printf("arb_xfrmode 0x%x, arb_xfrmodewait %d\n",
		      arb_xfrmode, arb_xfrmodewait);

	CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));
	CHECK(e1432_set_duty_cycle(hw, src_group, 1.0));
	CHECK(e1432_set_source_mode(hw, src_group, source_mode));
	CHECK(e1432_set_source_blocksize(hw, src_group, SRCBUFRSIZE));
	CHECK(e1432_set_ramp_rate(hw, src_group, 0));
	CHECK(e1432_set_range(hw, src_group, 1.0));
	CHECK(e1432_set_amp_scale(hw, src_group, .9995));

        /** create signal arrays ***/
	for (i=0; i < src_count; i++)
	{
	  src_sig(arb_siglength,2*arb_sigperiod/(2*(i+1)),src_sigbuffptrs_a[i]);
	  src_sig(arb_siglength,2*arb_sigperiod,src_sigbuffptrs_b[i]);
	}

	/* initialize source buffers */
	for (i=0; i < src_count; i++)
	{
	   CHECK(e1432_set_srcbuffer_mode(hw, src_list[i], arb_mode));
	   CHECK(e1432_set_srcbuffer_size(hw, src_list[i], arb_srcbufsize));
	   CHECK(e1432_set_srcbuffer_init(hw, src_list[i],
				       E1432_SRCBUFFER_INIT_EMPTY));
	}

	/* down load arb source signal */
	arb_xfrsize2 = arb_xfrsize;
	for (k=0; k < arb_srcbufsize; k = k + arb_xfrsize)
        {
	   arb_srcbufleft = arb_srcbufsize - k;
	   if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
	   for (i = 0; i < src_count; i++)
	   {
              error = e1432_write_srcbuffer_data(hw, src_list[i], 
			src_sigbuffptrs_a[i]+k, (SHORTSIZ16)  arb_xfrsize2, 
			arb_xfrmodewait);
              if(error)
                 printf("Error %d in e1432_write_srcbuffer_data\n", error);
	      printf("preload buffer A, chan %x, %d words\n", src_list[i],
			arb_xfrsize2);
	   }
	}

	if (arb_xfrmode == E1432_SRC_DATA_MODE_AB)
	{
	   printf("preload buffer B\n");
	   arb_xfrsize2 = arb_xfrsize;
	   for (k = 0; k < arb_srcbufsize; k = k + arb_xfrsize)
	   {
		arb_srcbufleft = arb_srcbufsize - k;
	        if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
		for (i = 0; i < src_count; i++)
		{
		   printf("preload buffer B, chan %x, %d words\n", src_list[i],
			arb_xfrsize2);
                   error = e1432_write_srcbuffer_data(hw, src_list[i], 
			src_sigbuffptrs_b[i]+k, (SHORTSIZ16)  arb_xfrsize2, 
			arb_xfrmodewait);
                   if(error)
                      printf("Error %d in e1432_write_srcbuffer_data\n", error);
		}
	   }
	}
	printf("done pre-loading buffer(s)\n");

	/* Set up interrupts */
	CHECK(irq_setup(laddr, &line));
	CHECK(e1432_set_interrupt_priority(hw, src_group, line));
        if(!arb_polling)
           CHECK(e1432_set_interrupt_mask(hw, src_group,
				       E1432_IRQ_BLOCK_READY |
				       E1432_IRQ_SRC_STATUS));
    }
    else
    {
	/* Set up interrupts */
	CHECK(irq_setup(laddr, &line));
	CHECK(e1432_set_interrupt_priority(hw, group, line));
	CHECK(e1432_set_interrupt_mask(hw, group,
				       E1432_IRQ_BLOCK_READY |
				       E1432_IRQ_SRC_STATUS));

	CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    }


    /* Start measurement */
    (void) printf("starting measurement\n");
    CHECK(e1432_init_measure(hw, global_group));

    /* Sleep for a long time.  Note that receiving an interrupt
       aborts the sleep, so we must keep calling it. */
    for (;;)
    {
	if (arb_polling)
	    (void) srcstatus();
	else
	    (void) sleep(1);
    }

    /* NOTREACHED */
    return 0;
}
